﻿using System.Collections.Generic;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Helpers
{
    public static class ValidationHelper
    {
        public static IEnumerable<string> Validate(Provider provider, bool forVaNetwork = false)
        {
            var messages = new List<string>();
            var providerType = NpiType.Item1Individual;

            // Required validations
            // Check provider id
            if (string.IsNullOrEmpty(provider.ProviderId))
            {
                messages.Add("Provider ID is required.");
            }

            // Check for VA network
            if (!forVaNetwork)
            {
                // Optional validations based on network
                // Check NPI
                if (provider.Npis != null && provider.Npis.Item.Count > 0)
                {
                    var npi = provider.Npis.Item[0];
                    if (!npi.Number.NpiCheck())
                    {
                        messages.Add($"NPI number ({npi.Number}) is invalid.");
                    }

                    // Gather provider type
                    providerType = npi.EntityTypeCode;
                }
                else
                {
                    messages.Add("NPI is required.");
                }

                // Check TIN
                if (provider.Tins != null && provider.Tins.Item.Count > 0)
                {
                    foreach (var item in provider.Tins.Item)
                    {
                        var tin = item.Number;
                        // Check number length
                        if (tin.Length != 9)
                        {
                            messages.Add($"Tax ID Number number ({tin}) is invalid.");
                        }
                    }
                }
                else
                {
                    messages.Add("At least one TIN is required.");
                }

                // Check service provider type
                if (!provider.HealthProviderTypeSpecified)
                {
                    messages.Add("Provider type is required.");
                }

                // Specialty
                if (provider.Specialties == null || provider.Specialties.Item.Count == 0)
                {
                    messages.Add("At least one provider specialty is required.");
                }
            }

            // Validate based on provider type
            switch (providerType)
            {
                case NpiType.Item1Individual:
                    messages.AddRange(Type1Check(provider, forVaNetwork));
                    break;
                default:
                    messages.AddRange(Type2Check(provider));
                    break;
            }

            return messages;
        }

        public static IEnumerable<string> Validate(DeaSchedulePrivilege privilege)
        {
            var messages = new List<string>();

            // Check DEA number
            if (privilege.DeaNumber != null)
            {
                if (!privilege.DeaNumber.DeaCheck())
                {
                    messages.Add($"DEA number ({privilege.DeaNumber}) is invalid.");
                }
            }
            else
            {
                messages.Add("DEA is required.");
            }

            return messages;
        }

        private static IEnumerable<string> Type1Check(Provider provider, bool forVaNetwork)
        {
            var messages = new List<string>();

            if (!(provider.Type.Item is Individual individual))
            {
                messages.Add("Invalid provider type.");
            }
            else
            {
                // Provider name
                if (string.IsNullOrEmpty(individual.FirstName) || string.IsNullOrEmpty(individual.LastName))
                {
                    messages.Add("The first and last name of the individual provider is required.");
                }

                // Provider Service
                if (!forVaNetwork && (individual.ProviderServices == null || individual.ProviderServices.Item.Count == 0))
                {
                    messages.Add("At least one provider service is required.");
                }
            }

            return messages;
        }

        private static IEnumerable<string> Type2Check(Provider provider)
        {
            var messages = new List<string>();
            if (!(provider.Type.Item is Organization org))
            {
                messages.Add("Invalid provider type.");
            }
            else
            {
                // Provider name
                if (string.IsNullOrEmpty(org.ProviderName))
                {
                    messages.Add("The Provider Name is required.");
                }
            }

            return messages;
        }
    }
}